(***********************************************************************)
(*                           ocamldsort                                *)
(*                                                                     *)
(*                 Copyright (C) 2002  Dimitri Ara                     *)
(*                                                                     *)
(* This program is free software; you can redistribute it and/or       *)
(* modify it under the terms of the GNU General Public License         *)
(* as published by the Free Software Foundation; either version 2      *)
(* of the License, or (at your option) any later version.              *)
(*                                                                     *)
(* This program is distributed in the hope that it will be useful,     *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of      *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       *)
(* GNU General Public License for more details.                        *)
(*                                                                     *)
(* You should have received a copy of the GNU General Public License   *)
(* along with this program; if not, write to the Free Software         *)
(* Foundation, Inc.,                                                   *)
(* 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.           *)
(***********************************************************************)

let load_path = ref []
let preprocessor = ref None
let files = ref []
let ocamldep = ref "ocamldep"

let add_file file =
  files := Filename.basename file :: !files

let usage = 
  "Usage: ocamldsort [-I <dir>] [-pp <command>] [-d <command>] <files>"

let is_input_a_tty () =
  try
    ignore (Unix.tcgetattr Unix.stdin);
    true
  with _ ->
    false

let speclist = [
  "-I", Arg.String (fun dir -> load_path := !load_path @ [dir]),
  "<dir>  Add <dir> to the list of include directories";
  "-pp", Arg.String (fun s -> preprocessor := Some s),
  "<command>  Pipe sources through preprocessor <command>";
  "-d", Arg.String (fun s -> ocamldep := s),
  "<command>  Use <command> to do ocamldep job";
]

let fatal_error str =
  prerr_endline str;
  exit 2

let _ =
  try
    Arg.parse speclist add_file usage; 
    let input_param =
      if is_input_a_tty () then
	Params.new_ocamldep_input_param !ocamldep !load_path !preprocessor
    else 
      Params.Stdin in
    let param = Params.new_param input_param !files in
    let deps = Dependencies.get_dependencies param in
    let sorted = Dep_sort.sort_dependencies deps param in
      (match sorted with
	| [] -> ()
	| sorted -> print_endline (String.concat " " sorted))
  with 
    | Dep_sort.Dependency_error _ ->
	fatal_error "Error: dependency error (cyclic dependencies?)"
    | Files.Unknown_extension file ->
	fatal_error ("Error: what kind of file " ^ file ^ " is?")
    | Dep_parse.Parse_error ->
	fatal_error "Error: error while parsing ocamldep output"
    | exc ->
	fatal_error ("Error: " ^ (Printexc.to_string exc))
