{-# LANGUAGE MultiParamTypeClasses #-}

module Theta where

import qualified Delta as D
import Delta (Mapping)

import ExtList
import Data.Maybe

--
-- Types
--

data Object = Zero | Wreath Int [Object]
              deriving (Eq, Ord, Show, Read)

data RawMorphism = MZero Int | MWreath Mapping [[RawMorphism]] 
                   deriving (Eq, Ord, Show, Read)

data Morphism = Morphism {
                  source :: Object,
                  target :: Object,
                  rawMorphism ::  RawMorphism
                } deriving (Eq, Ord, Show, Read)

--
-- Operations on Objects
--

dim :: Object -> Int
dim Zero         = 0
dim (Wreath n l) = foldl (\acc x -> acc + dim x) n l

isDim0 :: Object -> Bool
isDim0 Zero = True
isDim0 _    = False

height :: Object -> Int
height Zero         = 0
height (Wreath _ l) = 1 + foldl (\acc x -> max acc (height x)) 0 l

trunc1 :: Object -> Int
trunc1 Zero         = 0
trunc1 (Wreath n _) = n

cotrunc1 :: Object -> [Object]
cotrunc1 Zero         = []
cotrunc1 (Wreath _ l) = l

wreath :: Object -> Object
wreath o = Wreath 1 [o]

wreath2 :: Object -> Object -> Object
wreath2 o1 o2 = Wreath 2 [o1, o2]

wreath3 :: Object -> Object -> Object -> Object
wreath3 o1 o2 o3 = Wreath 3 [o1, o2, o3]

wreath4 :: Object -> Object -> Object -> Object -> Object
wreath4 o1 o2 o3 o4 = Wreath 4 [o1, o2, o3, o4]

suspend :: Object -> Object
suspend = wreath

suspendN :: Int -> Object -> Object
suspendN 0 o = o
suspendN n o = suspend (suspendN (n - 1) o)

--
-- Predefined Objects
--

disk :: Int -> Object
disk 0 = Zero
disk n = suspend (disk (n - 1))

delta :: Int -> Object
delta 0 = Zero
delta n = Wreath n (replicate n Zero)

--
-- Operations on morphisms
--

dimSubObj :: Morphism -> Int
dimSubObj = dim . source

suspendMorph (Morphism s t r) =
  Morphism (suspend s) (suspend t) (MWreath [0, 1] [[r]])

--
-- Predefined morphisms
--

rawMorphFromMap :: Maybe Int -> Mapping -> RawMorphism
rawMorphFromMap s f 
  | s' == 0 =  MZero (D.evalMapping f 0)
  | otherwise = MWreath f $ map (\n -> replicate n (MZero 0)) (diff f)
 where s' = fromMaybe (length f - 1) s

morphFromMap :: Int -> Int -> Mapping -> Morphism
morphFromMap s t f = Morphism (delta s) (delta t) (rawMorphFromMap (Just s) f)

-- assume n > 0
sourceDisk :: Int -> Morphism
sourceDisk 1 = Morphism (disk 0) (disk 1) (MZero 0)
sourceDisk n = suspendMorph $ sourceDisk (n - 1)

-- assume n > 0
targetDisk :: Int -> Morphism
targetDisk 1 = Morphism (disk 0) (disk 1) (MZero 1)
targetDisk n = suspendMorph $ targetDisk (n - 1)

--
-- Misc
--

splitWreath :: [Int] -> [a] -> [[a]]
splitWreath is l = tail $ splitSize (diff (0:is)) l

--
-- Composition
--

compRawMorphism :: RawMorphism -> RawMorphism -> RawMorphism
compRawMorphism (MZero i) (MZero _) = MZero i
compRawMorphism (MWreath f ms) (MZero i) = MZero (D.evalMapping f i)
compRawMorphism (MZero i) (MWreath f ms) = 
  MWreath (replicate len i) (replicate (len - 1) [])
  -- the cost of computing could be avoided if the source was provided
    where len = length f
compRawMorphism (MWreath g ns) (MWreath f ms) = MWreath (D.compMapping g f) ps
    -- to rewrite
    where ps = forZip (splitWreath f ns) ms
                 (\ n m -> concat $ forZip n m
                    (\ n' m' -> [compRawMorphism n'' m' | n'' <- n']))

compMorphism :: Morphism -> Morphism -> Morphism
compMorphism (Morphism _ t g) (Morphism s _ f) =
  Morphism s t (compRawMorphism g f)
