{-# LANGUAGE MultiParamTypeClasses, FunctionalDependencies #-}

module GlobParStruct where

import Prelude hiding (lookup)

import qualified Data.Map as Map
import qualified Data.List as List
import qualified Data.Set as Set
import Data.Maybe

import qualified FreeAbGrp as FAG
import FreeAbGrp (FreeAbGrp)
import ExtList

class (Ord o, Ord m) => DimCategory o m | o -> m, m -> o where
  source :: m -> o
  compose :: m -> m -> m
  dim :: o -> Int
  isDim0 :: o -> Bool
  isDim0 x = dim x == 0
  subObjsCodim1 :: o -> [m]

type Object o m = o
type Morphism o m = m
type SourcesTargets o m = ([m], [m])
type GlobParStruct o m = Map.Map o (SourcesTargets o m)
type CheckGPSFun o m = GlobParStruct o m -> o -> Bool
type CheckGPSFunST o m = GlobParStruct o m -> o -> SourcesTargets o m -> Bool

empty :: GlobParStruct o m
empty = Map.empty

member :: DimCategory o m => o -> GlobParStruct o m -> Bool
member = Map.member

insert :: DimCategory o m => Object o m -> SourcesTargets o m -> 
               GlobParStruct o m -> GlobParStruct o m
insert = Map.insert

-- unsafe: change?
lookup :: DimCategory o m =>
               Object o m -> GlobParStruct o m -> SourcesTargets o m
lookup = flip (Map.!)

lookupMorphism :: DimCategory o m =>
                    Morphism o m -> GlobParStruct o m -> SourcesTargets o m
lookupMorphism f gps =
  ([compose f g | g <- s'], [compose f g | g <- t'])
    where s = source f 
          (s', t') = lookup s gps

sourcesTargets :: DimCategory o m =>
                    [Morphism o m] -> GlobParStruct o m -> SourcesTargets o m
sourcesTargets ms gps =
  concatPair $ map (flip lookupMorphism $ gps) ms

sources :: DimCategory o m =>
             [Morphism o m] -> GlobParStruct o m -> [Morphism o m]
sources ms gps = fst $ sourcesTargets ms gps

targets :: DimCategory o m =>
             [Morphism o m] -> GlobParStruct o m -> [Morphism o m]
targets ms gps = snd $ sourcesTargets ms gps

--
-- axioms
--

(?&&) :: DimCategory o m =>
           CheckGPSFunST o m -> CheckGPSFunST o m -> CheckGPSFunST o m
f ?&& g = \gps o st -> f gps o st && g gps o st

(??&&) :: DimCategory o m =>
            CheckGPSFun o m -> CheckGPSFun o m -> CheckGPSFun o m
f ??&& g = \gps o -> f gps o && g gps o


-- CP1 "dijoint"

setDisjoint x y = Set.null $ x `Set.intersection` y

checkCP1ST :: DimCategory o m => CheckGPSFunST o m
checkCP1ST gps _ (s, t) = 
  let (ss, ts) = sourcesTargets s gps
      (st, tt) = sourcesTargets t gps
      [ss', ts', st', tt'] = map Set.fromList [ss, ts, st, tt]
  in  setDisjoint ss' tt' && setDisjoint ts' st' &&
        (ss' `Set.union` tt') == (ts' `Set.union` st')

{-
  let (ss, ts) = sourcesTargets s gps
      (st, tt) = sourcesTargets t gps
  in  disjoint ss tt && disjoint ts st &&
        (ss `List.union` tt) `eqAsSet` (ts `List.union` st)
-}

checkCP1 :: DimCategory o m => GlobParStruct o m -> Object o m -> Bool
checkCP1 gps o = checkCP1ST gps o (lookup o gps)

-- CP2

wellFormed :: DimCategory o m => GlobParStruct o m -> [Morphism o m] -> Bool
wellFormed gps [] = True
wellFormed gps ms@(m:ms')
  | isDim0 (source m) = null ms'
  | otherwise             =
      and $ do (x, y) <- distPairs ms
               let (sx, tx) = lookupMorphism x gps
                   (sy, ty) = lookupMorphism y gps
               return $ disjoint sx sy && disjoint tx ty

{-
wellFormed :: DimCategory o m => GlobParStruct o m -> [Morphism o m] -> Bool
wellFormed gps [] = False
wellFormed gps ms @(m:ms')
  | isDim0 m = not (null ms')
  | otherwise  =
      and $ do (x, y) <- distPairs ms
               let (sx, tx) = lookupMorphism x gps
                   (sy, ty) = lookupMorphism y gps
                   [sx', tx', sy', ty'] = map Set.fromList [sx, tx, sy, ty]
               return $ setDisjoint sx' sy' && setDisjoint tx' ty'
-}

checkCP2ST :: DimCategory o m => CheckGPSFunST o m
checkCP2ST gps _ (s, t) =
  wellFormed gps s && wellFormed gps t

checkCP2 :: DimCategory o m => GlobParStruct o m -> Object o m -> Bool
checkCP2 gps o = checkCP2ST gps o (lookup o gps)

-- iterated diffs

diffST :: DimCategory o m =>
            GlobParStruct o m -> SourcesTargets o m -> FreeAbGrp (Morphism o m)
diffST gps (s, t) = (FAG.fromList t) - (FAG.fromList s)

diffMorph :: DimCategory o m =>
               GlobParStruct o m -> Morphism o m -> FreeAbGrp (Morphism o m)
diffMorph gps m = diffST gps (s, t)
  where (s, t) = lookupMorphism m gps

diffMorphs :: DimCategory o m => GlobParStruct o m ->
                FreeAbGrp (Morphism o m) -> FreeAbGrp (Morphism o m)
diffMorphs gps x = x FAG.?>>= diffMorph gps

diffM :: DimCategory o m => GlobParStruct o m -> FreeAbGrp (Morphism o m) ->
           FreeAbGrp (Morphism o m)
diffM gps = FAG.negPart . diffMorphs gps

diffP :: DimCategory o m => GlobParStruct o m -> FreeAbGrp (Morphism o m) ->
           FreeAbGrp (Morphism o m)
diffP gps = FAG.posPart . diffMorphs gps

iterDiffST :: DimCategory o m => 
                GlobParStruct o m -> Object o m -> SourcesTargets o m ->
                ([FreeAbGrp (Morphism o m)], [FreeAbGrp (Morphism o m)])
iterDiffST gps o (s, t) = (take d $ iterate (diffM gps) (FAG.fromList s),
                         take d $ iterate (diffP gps) (FAG.fromList t))
                           where d = dim o

iterDiff :: DimCategory o m => GlobParStruct o m -> Object o m -> 
              ([FreeAbGrp (Morphism o m)], [FreeAbGrp (Morphism o m)])
iterDiff gps o = iterDiffST gps o (lookup o gps)

containsOnly1 :: FreeAbGrp a -> Bool
containsOnly1 = all (==1) . FAG.coeffs

checkDiffIt :: [FreeAbGrp a] -> Bool
checkDiffIt [] = True
checkDiffIt [ms] = containsOnly1 ms && length1 (FAG.support ms)
checkDiffIt (ms:mms) = containsOnly1 ms && checkDiffIt mms 

checkDST :: DimCategory o m => CheckGPSFunST o m
checkDST gps o (s, t) =
  checkDiffIt dpit && checkDiffIt dmit
    where (dpit, dmit) = iterDiffST gps o (s, t)

checkD :: DimCategory o m => GlobParStruct o m -> Object o m -> Bool
checkD gps o = checkDST gps o (lookup o gps)

checkD2ST :: DimCategory o m => CheckGPSFunST o m
checkD2ST gps o (s, t)
  | isDim0 o = True
  | otherwise = (diffMorphs gps . diffST gps $ (s, t)) == FAG.zero

checkD2 :: DimCategory o m => GlobParStruct o m -> Object o m -> Bool
checkD2 gps o = checkD2ST gps o (lookup o gps)

checkAllST :: DimCategory o m => CheckGPSFunST o m
checkAllST = checkCP2ST ?&& checkDST ?&& checkCP1ST

checkAll :: DimCategory o m => CheckGPSFun o m
checkAll = checkCP2 ??&& checkD ??&& checkCP1

checkWhole :: DimCategory o m => GlobParStruct o m -> Bool
checkWhole gps = and $ map (checkAll gps) (Map.keys gps)

--
-- main loop
--

getNewSTs :: DimCategory o m => Object o m -> GlobParStruct o m -> 
               CheckGPSFunST o m -> [SourcesTargets o m]
getNewSTs x gps f =
  filter (f gps x) $
    if member x gps 
      then [lookup x gps]
      else biPartitions (subObjsCodim1 x)

-- assume that the objects are "sorted according to faces"
generate :: DimCategory o m => [Object o m] -> [Object o m] ->
             [GlobParStruct o m] -> CheckGPSFunST o m -> [GlobParStruct o m]
generate []     []     gpss _ = gpss
generate _      _      []   _ = []
generate (x:xs) ys     gpss f = generate xs ys newGpss f
  where newGpss = do
          gps <- gpss
          st <- getNewSTs x gps f
          return $ insert x st gps
generate []     (y:ys) gpss f = generate [] ys newGpss f
  where newGpss = do
          gps <- gpss
          if null $ getNewSTs y gps f
            then []
            else [gps]
