(**************************************************************************)
(*                                                                        *)
(*                                 Averell                                *)
(*                                                                        *)
(*          Vincent Simonet, Projet Cristal, INRIA Rocquencourt           *)
(*                                                                        *)
(*  Copyright 2002 Institut National de Recherche en Informatique et en   *)
(*  Automatique.  All rights reserved.  This file is distributed under    *)
(*  the terms of the GNU Library General Public License, with the         *)
(*  special exception on linking described in file LICENSE.               *)
(*                                                                        *)
(*  Author contact: Vincent.Simonet@inria.fr                              *)
(*  Software page: http://cristal.inria.fr/~simonet/soft/                 *)
(*                                                                        *)
(**************************************************************************)

(* $Id: avl_tarjan.ml,v 1.1 2002/10/29 15:46:32 simonet Exp $ *)

(** Tarjan's algorithm: calculating SCC of a graph in linear time.

    This module provides an implementation of Tarjan's algorithm.  This
    algorithm computes the strong connex composants of a directed graph
    (SCC).  A SCC of a graph $G = (X, E)$ is a subset of $X$ such that for
    every pair of nodes $x_1$ and $x_2$ in $X$ there exists a path from
    the former to the latter in $G$.  The SCCs of $G$ form a partition of 
    $X$.  

    The Tarjan's algorithm has a time complexity in $O(n)$ (where $n$ is the
    number of nodes of the input graph).

    Functions provided by this module are not reentering thread safe as long 
    as at most one thread operates on the same graph at the same time.
 *)



(** The client must provide an implementation of graphs which fullfills the
    signature [GRAPH]. 
 *)
module type GRAPH = sig

  (** The type of graphs. *)
  type graph

  (** The type of nodes. *)
  type node

  (** [iter_nodes f g] applies [f] on every nodes of the graph [g].  The 
      order in which nodes are considered does not matter and may change
      between different application of the function on the same graph.  
      However, each node must be considered exactly once.
   *)
  val iter_nodes: (node -> unit) -> graph -> unit

  (** [iter_successors f nd] applies [f] on every successors of the
      node [nd] in its graph.  The order in which successors are
      considere does not matter.  The graph is not required to be
      simple (i.e. [iter_successors f nd] may apply [f] an arbitrary
      number of time the function [f] on each of [nd]'s successors, as
      long as this number is fixed.
   *)
  val iter_successors: (node -> unit) -> node -> unit

  (** Every node must carry a transient integer field.  The following
      functions allows reading and updating this field.  No assumption
      is made by the module on the initial content of this field at
      each function call.  However, the client cannot make any
      assumption on the final content too.
   *) 
  val get: node -> int
  val set: node -> int -> unit

end



(* The internal integer fields of nodes are used to store three informations:
   - a boolean (bit 0)
   - the number given to the node (bits 1 to 15)
   - the attache number of the node (bits 16 to 31)
 *)

let stacked_mask = 0b0000000000000000000000000000001
let stacked_comp = lnot stacked_mask
let stacked_shift = 0
let number_mask =  0b0000000000000001111111111111110
let number_comp = lnot number_mask
let number_shift = 1
let attache_mask = 0b1111111111111110000000000000000
let attache_comp = lnot attache_mask
let attache_shift = 16



module Make (X : GRAPH) = struct


  (** [fold empty add_class singleton add g] runs the Tarjan's algorithm
      on the graph [g].  It returns a partition of the set of the nodes 
      of the graph (i.e. a set of set of nodes), which is computed as follows:
      - [empty] is the empty partition,
      - [add_class c p] adds the class [c] to the partition [p],
      - [singleton nd] returns the class with one element [nd],
      - [add nd c] add the node [nd] to the class [c].
   *)
  let fold empty add_set singleton add graph =

    (* We ensure that number and attache fields of each node
       are equal to 0. *)

    X.iter_nodes (function nd -> X.set nd 0) graph;

    (* We define the stack used by Tarjan's algorithm to store SCC.
       This stack contains elements of type flesh_proxy. The two functions
       push and pop a flesh on the stack and update their stacked field *)

    let tarjan_stack = Stack.create () in

    let pop () =
      let nd = Stack.pop tarjan_stack in
      X.set nd (X.get nd land stacked_comp);
      nd
    in

    (* Following the Tremaux's algorithm, we will number nodes. 
       mark set a fresh number to a node. *)

    let counter = ref 0 in

    (* The resulting list. *)

    let result = ref empty in
    
    let rec walk nd =

      (* The current receive a fresh number and is stacked. *)

      incr counter;
      Stack.push nd tarjan_stack;
      X.set nd (stacked_mask
		  lor (!counter lsl attache_shift) 
		  lor (!counter lsl number_shift));

      (* Then we visit its successors *)

      X.iter_successors (function nd' ->
	
	if X.get nd' land number_mask = 0 then begin
	  walk nd';
	  let i = X.get nd 
	  and att' = (X.get nd') land attache_mask in
	  if att' < i land attache_mask
	  then X.set nd ((i land attache_comp) lor att')
	end
	else begin
	  let i = X.get nd
	  and i' = X.get nd' in
	  let num' = i' land number_mask in

	  if num' < i land number_mask && i' land stacked_mask <> 0
	  then X.set nd 
	      (i land attache_comp lor (num' lsl (attache_shift - number_shift)))
	end
	) nd;
      
      (* If [node.attache = node.number] then [node] is the root of an SCC *)

      let i = X.get nd in
      if (i land number_mask) lsl (attache_shift - number_shift)
	  = i land attache_mask
      then begin
	let set = ref (singleton nd) in
	while Stack.top tarjan_stack != nd do (* TEMPORARY UTILISER LES NUMBER*)
	  set := add (pop ()) !set
	done;
	ignore (pop ());
	result := add_set !set !result
      end
    in

    (* Start point of Tarjan's algorithm *)

    X.iter_nodes (function nd ->
      if X.get nd land number_mask = 0 then walk nd
    ) graph;

    ! result



  (** [list g] computes the SCCs of a graph.  The result is a list of list
      of nodes: each list gives the nodes of one of the SCCs.
   *)
  let list graph =

    fold
      [] (fun c p -> c :: p)
      (fun nd -> [nd]) (fun nd c -> nd :: c)
      graph



  (** [unify unifier graph] computes the SCCs of a graph.  For every SCC,
      it chooses a particular node [nd], and for every other node [nd'] of
      the SCC, [unifier nd nd'] is called.
   *)
  let unify unifier graph =

    fold
      () (fun _ _ -> ())
      (fun nd -> nd) (fun nd' nd -> unifier nd nd'; nd)
      graph

end
