(**************************************************************************)
(*                                                                        *)
(*                    APM interface for Objective Caml                    *)
(*                                                                        *)
(*           Vincent Simonet, projet Cristal, INRIA Rocquencourt          *)
(*                                                                        *)
(*  Copyright 2002                                                        *)
(*  Institut National de Recherche en Informatique et en Automatique      *)
(*  All rights reserved.  This file is distributed under the terms of     *)
(*  the GNU Library General Public License (see file LICENSE).            *)
(*                                                                        *)
(*  Vincent.Simonet@inria.fr        http://cristal.inria.fr/~simonet/     *)
(*                                                                        *)
(**************************************************************************)

(* $Id$ *)

open Printf
open Apm



(***************************************************************************)

let version = "1.00"
let verbose = ref false
let minutes = ref false



let print_version () =
  printf "mlapm version %s\n" version;
  exit 0



(***************************************************************************)

let main () =

  (* Checking APM support *)

  begin match exists () with
    Supported -> ()
  | Unsupported ->
      eprintf "No APM support in kernel\n";
      exit 1
  | WrongVersion ->
      eprintf "Old APM support in kernel\n";
      exit 2
  end;

  (* Parsing arguments *)

  Arg.parse [

  "-V", Arg.Unit print_version,
  "Print program version and exit immediately";

  "-v", Arg.Set verbose,
  "Print information about the APM BIOS version and Linux APM driver version.";

  "-m", Arg.Set minutes,
  "Print total minutes remaining instead of using an hh:mm format.";

  "-s", Arg.Unit suspend,
  "Put the machine into suspend mode if possible.";

  "-S", Arg.Unit standby,
  "Put the machine into standby mode if possible." ] 

  (function _ -> ())

  "Usage: mlapm <options>\nOptions are:";

  let i = read () in

  if !verbose then
    printf "APM BIOS %d.%d (kernel driver %s)\n"
      i.version_major i.version_minor i.driver_version;

  if not i.support_32_bits then begin
    eprintf "32-bit APM interface not supported\n";
    exit 1
  end;

  if !verbose && i.power_management_disabled then
    printf "APM BIOS Power Management is currently disabled\n";
  if !verbose && i.power_management_disengaged then
    printf "APM BIOS Power Management is currently disengaged\n";

  begin match i.ac_line_status with
    OffLine -> printf "AC off-line"
  | OnLine -> printf "AC on-line"
  | OnBackupPower -> printf "On backup power"
  end;

  begin match i.battery_status with
    High -> printf ", battery status high"
  | Low -> printf ", battery status low"
  | Critical -> printf ", battery status critical"
  | Charging -> printf ", battery charging"
  | NoBattery -> printf ", no system battery"
  end;

  begin match i.battery_percentage with
    None -> ()
  | Some d -> printf ": %d%%" d
  end;

  begin match i.battery_time with
    None -> ()
  | Some secs ->
      let m = secs / 60 in
      if !minutes then printf " (%d min)" m
      else begin
	let d = m / 1440 in
	let h = (m / 60) mod 24 in
	let m = m mod 1440 in
	if d > 1 then 
	  printf " (%d day%s %d:%02d)" d (if d = 1 then "" else "s") h m
	else
	  printf " (%d:%02d)" h m
      end
  end;

  printf "\n";
  ()



let () =
  handle_apm_error main ()
