/**************************************************************************/
/*                                                                        */
/*                    APM interface for Objective Caml                    */
/*                                                                        */
/*           Vincent Simonet, projet Cristal, INRIA Rocquencourt          */
/*                                                                        */
/*  Copyright 2002                                                        */
/*  Institut National de Recherche en Informatique et en Automatique      */
/*  All rights reserved.  This file is distributed under the terms of     */
/*  the GNU Library General Public License (see file LICENSE).            */
/*                                                                        */
/*  Vincent.Simonet@inria.fr        http://cristal.inria.fr/~simonet/     */
/*                                                                        */
/**************************************************************************/


#include <caml/mlvalues.h>
#include <caml/alloc.h>
#include <caml/memory.h>
#include <caml/callback.h>

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <apm.h>
#include <time.h>

void camlapm_failure (char * msg1, char * msg2) 
{
  callback2 (*caml_named_value ("Apm.failure"), 
	     copy_string(msg1), 
	     copy_string(msg2));
}

CAMLprim value camlapm_exists (value unit)
{
  CAMLparam1 (unit);
  CAMLreturn (Val_int (apm_exists ()));
}

CAMLprim value camlapm_read (value unit)
{
  CAMLparam1 (unit);
  CAMLlocal1 (r);

  apm_info i;
  
  switch (apm_read (&i))
    { 
    case 0:
      r = alloc (10, 0);

      Store_field (r, 0, copy_string(i.driver_version));
      Store_field (r, 1, Val_int(i.apm_version_major));
      Store_field (r, 2, Val_int(i.apm_version_minor));

      /* 32 bit support */
      Store_field (r, 3, Val_bool (i.apm_flags & APM_32_BIT_SUPPORT));

      /* Power Management disabled */
      Store_field (r, 4, Val_bool (i.apm_flags & 0x10));

      /* Power Management disengaged */
      Store_field (r, 5, Val_bool (i.apm_flags & 0x20));

      /* AC line status */
      Store_field (r, 6, Val_int (i.ac_line_status));

      /* Battery status */
      if (i.battery_flags != 0xff && i.battery_flags & 0x80)
	{ Store_field (r, 7, 4); }
      else { 
	switch (i.battery_status) { 
	case 0: Store_field (r, 7, Val_int(0)); break;
	case 1: Store_field (r, 7, Val_int(1)); break;
	case 2: Store_field (r, 7, Val_int(2)); break;
	case 3: Store_field (r, 7, Val_int(3)); break;
	default: Store_field (r, 7, Val_int(4)); break;
	}
      }

      if (i.battery_percentage < 0)
	{ Store_field (r, 8, Val_int(0)); }
      else {
	CAMLlocal1(opt);
	opt = alloc_small(1, 0);
	Field(opt, 0) = Val_int(i.battery_percentage);
	Store_field (r, 8, opt);
      }

      if (i.battery_time < 0)
	{ Store_field (r, 9, Val_int(0)); }
      else {
	CAMLlocal1(opt);
	opt = alloc_small(1, 0);
	Field(opt, 0) = 
	  Val_int(i.using_minutes ? i.battery_time * 60 : i.battery_time);
	Store_field (r, 9, opt);
      }

      break;
    case 1:
      camlapm_failure("Cannot read APM information", "no kernel support");
      break;
    case 2:
      camlapm_failure
	("Cannot read APM information", "wrong version of kernel support");
      break;
    }

  CAMLreturn (r);

}

void camlapm_change_state (int mode, char * msg)
{

  int fd;
  time_t then, now;
  int error;

  sync();

  if ((fd = apm_open()) < 0)
    camlapm_failure ("Cannot open APM device", strerror(errno));

  time(&then);

  switch (mode)
    {
    case SUSPEND:
      error = apm_suspend(fd);
      break;
    case STANDBY:
      error = apm_standby(fd);
      break;
#ifdef APM_IOC_IGNORE /* detect kernel support of IGNORE/NOIGNORE functions */
    case IGNORE:
    case NOIGNORE:
      error = apm_set_ignore(fd, mode);
      apm_close(fd);
      return;
      break;
#endif
    default:
      error = 0;
      break;
    }
    
  time(&now);
  apm_close(fd);

  if (error != 0)
    camlapm_failure (msg, strerror(errno));

}



CAMLprim value camlapm_suspend (value unit)
{
  CAMLparam1 (unit);
  camlapm_change_state (SUSPEND, "Suspend fails");
  CAMLreturn (Val_int(0));
}



CAMLprim value camlapm_standby (value unit)
{
  CAMLparam1 (unit);
  camlapm_change_state (STANDBY, "Standby fails");
  CAMLreturn (Val_int(0));
}
