(**************************************************************************)
(*                                                                        *)
(*                    APM interface for Objective Caml                    *)
(*                                                                        *)
(*           Vincent Simonet, projet Cristal, INRIA Rocquencourt          *)
(*                                                                        *)
(*  Copyright 2002                                                        *)
(*  Institut National de Recherche en Informatique et en Automatique      *)
(*  All rights reserved.  This file is distributed under the terms of     *)
(*  the GNU Library General Public License (see file LICENSE).            *)
(*                                                                        *)
(*  Vincent.Simonet@inria.fr        http://cristal.inria.fr/~simonet/     *)
(*                                                                        *)
(**************************************************************************)



(***************************************************************************)
(** {2 Errors report} *)

(** Functions of this library raise an exception [Failure] with an
    appropriate message when they encouter an error.
 *)
exception Failure of string * string

let failure msg1 msg2 =
  raise (Failure (msg1, msg2))

let _ =
  Callback.register "Apm.failure" failure



(** [handle_apm_error f x] applies [f] to [x] and returns the result.
    If the exception [Failure] is raised, it prints a message
    describing the error and exits with code 2.
 *)
let handle_apm_error f x =
  try
    f x
  with
    Failure (msg1, msg2) ->
      prerr_string Sys.argv.(0);
      prerr_string ": APM failure (";
      prerr_string msg1;
      if msg2 <> "" then (prerr_string ": "; prerr_string msg2);
      prerr_endline ")";
      exit 2



(***************************************************************************)
(** {2 APM status} *)

type apm_support =
    Supported
      (** APM support of the right version exists in the kernel. *)
  | Unsupported
      (** No APM support in the kernel. *)
  | WrongVersion
      (** APM support of a wrong version exists in the kernel. *)

(** [exists ()] returns a flag indicating the status of the support of
    APM in the kernel.
 *)
external exists: unit -> apm_support = "camlapm_exists"



(***************************************************************************)
(** {2 Getting APM info} *)

(** The AC line status is reported using a value of type [ac_line_status].
 *)
type ac_line_status =
    OffLine
      (** AC off-line *)
  | OnLine
      (** AC on-line *)
  | OnBackupPower
      (** On backup power *)

(** The battery status is reported using a value of type [battery_status].
 *)
type battery_status =
    High
      (** Battery status high *)
  | Low
      (** Battery status low *)
  | Critical
      (** Battery status critical *)
  | Charging
      (** Battery charging *)
  | NoBattery
      (** No battery. *)
    
(** Information about APM are reported using a record of type [info].
    It provides generic information about the driver and current information
    about the AC line and battery status.
 *)
type info =
    { driver_version: string;
        (** The kernel driver version. *)
      version_major: int;
        (** The major version number of the APM BIOS. *)
      version_minor: int;
        (** The minor version number of the APM BIOS. *)
      support_32_bits: bool;
        (** Indicates wether the 32-bit APM interface is supported. *)
      power_management_disabled: bool;
        (** Indicates wether the APM BIOS Power Management is currently 
	    disabled. *)
      power_management_disengaged: bool;
        (** Indicates wether the APM BIOS Power Management is currently
	    disengaged. *)
      ac_line_status: ac_line_status;
        (** Gets the current AC line status. *)
      battery_status: battery_status;
        (** Gets the current battery status. *)
      battery_percentage: int option;
        (** The percentage of charging of the battery, if available. *)
      battery_time: int option
	(** The time in seconds, if available. *)
    } 


(** [read ()] returns a record of type [info] giving the current information
    from the APM driver.  If APM is not supported by the kernel, an exception
    [!Failure] is raised.
 *)
external read: unit -> info = "camlapm_read"



(***************************************************************************)
(** {2 Sending messages} *)

(** [apm_suspend ()] tries to set the power state to suspend. 
 *)
external suspend: unit -> unit = "camlapm_suspend"

(** [apm_standby ()] tries to set the power state to standby. 
 *)
external standby: unit -> unit = "camlapm_standby"
